<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-as-a-service
* @subpackage models
* @filesource
*/

#TODO - HAS RELATED, ADD RELATED
#TODO - MOVE SELECTS TO FIND_RELATED - OR - SIMPLIFY SO THAT IN GENERAL THAT WE DON'T USE JOINS AND DON'T BY DEFAULT SELECT ANYTHING

/** */
require_once BASEPATH.'core/Model.php';

/**
* Parent class for models based on a database table.
*
* Handles basic CRUD functionality, reducing the amount of custom code that needs to be written for each model & enforcing consistency.
* This parent classed is based off of the {@link http://en.wikipedia.org/wiki/Active_record_pattern Active Record Pattern}.  Static
* methods deal with multiple rows of the database table, or meta-information about the model as whole.  An instance of this model
* encapsulates a specific row of the database and useful methods for dealing with that data, including the ability to retrieve 
* related entities.
*
* *NOTE* - Models that extend from this class deviate from the usual CI model paradigm and should *NOT* be loaded in the usual CI fashion.
* The CI model loader instantiates a single instance of the model class and adds it as a class variable to the controller so that its methods
* are available throughout the application; this was a work around to cope with PHP's limited static inheritance capabilities prior to PHP 5.3.
* Since the paradigm for this model assumes that each instance of the model encapsulates a row of the database, attempting to load the model
* will only confuse the issue.
*
*
* @package direct-as-a-service
* @subpackage models
*/
class Entity extends CI_Model {
	
	/////////////////////////////////
	// STATIC VARS
	////////////////////////////////
	
	/** @var string */
	static $database_group = 'default';
	
	/** @var string */
	static $table;
	
	/** @var string */
	static $primary_key = 'id'; //set this in child class if the primary key is something else
	
	/** @var array */
	protected static $_relationships = array(); //protected because we merge it with its parents; call on relationships() instead;
	
	////////////////////////////////
	// INSTANCE VARS
	////////////////////////////////

	/** @var CI_DB */
	public $db;
	
	public $model_alias;
	
	/** @var Validator */
	protected $error;
	
	/** @var Error_helper */
	protected $is;
	
	/** 
	* The database record for this entity as an array formatted array($field => $value).
	*
	* Note that this array will reflect the results of {@link find()}, which may include fields that are not columns on the table 
	* (for example, the result of a calculation in the query, or a field from a joined table). 
	*
	* All of the fields included in this array are accessible as class properties, so that they may be accessed like this:
	*
	* <code>
	* //if the table has a field called 'weasley' ...
	* echo $entity->weasley; //same as echo $entity->_values['weasley'];
	* $entity->weasley = 'fred'; //same as $entity->_values['weasley'] = 'fred';
	* $entity->save(); //saves the new value to the database
	* </code>
	*
	* @var array */
	protected $_values = array();
	protected $_altered_fields = array();
	
	
	/** 
	* Fields on this model that cannot be written to the database, e.g. the primary key, auto-generated timestamps, etc.
	* This may also include fields generated by a database query that do not represent a column of the database table 
	* (for example, the result of a calculation in the query, or a field from a joined table).
	* Do not access this variable directly - use {@link readonly_fields()} as an accessor.
	* @var array */
	protected $_readonly_fields = array(); 	
	
	protected $_validation_rules = array();
	
	protected $_property_validation_rules = array('property_validation_rules' => 'array');
	
/////////////////////////////////////////////////////////////////////////////////////
// INSTANCE METHODS 
// Methods concerning a single row in the database table.
//////////////////////////////////////////////////////////////////////////////////////
	
	public function __construct($values = array()){			
		if(is_a($this, 'Entity') && get_class($this) != 'Entity'){			
			$CI = get_instance();
		
			$this->db = static::db();
			$this->error = $CI->error;
			$this->is = $CI->is;
			$this->model_alias = static::model_alias();

			$table = static::$table; //not nesc., but makes phpdocumentor happier
			$primary_key = static::$primary_key; //not nesc., but makes phpdocumentor happier

			//verify that the table & primary_key values are valid
			if(empty($table) || !$this->db->table_exists($table)){
				$this->error->should_be_a_table_in_the_database($table);
				#show_error(DEFAULT_ERROR_MESSAGE);
			}
			if(empty($primary_key) || !$this->db()->field_exists($primary_key, $table)){
				$this->error->should_be_a_column_for_table($primary_key, $table);
				#show_error(DEFAULT_ERROR_MESSAGE);
			}		
			
			if(!is_array($values)) return $this->error->should_be_an_array($values);
			if(!empty($values)){
				$this->set_values($values);
			}
			
			//once we've initialized the values, don't let people change the value of the primary key
			$this->_readonly_fields[] = static::$primary_key;
		} 
	}
		
	
	/** @return int|null */ 
	public function id(){
#TODO - interesting that past-me specified null rather than false.  double-check magical methods to see if this affects magical id() vs id, and if I want to go back to that.		
		return element(static::$primary_key, $this->_values, NULL);
	}
	
	public function altered_values(){
		return $this->values($this->altered_fields);
	}	
	
	public function describe(){
		$model_alias = static::model_alias();
		if($this->property_is_empty(static::$primary_key)) return 'this unsaved '.$model_alias;
		return $model_alias.'#'.$this->id;
	}		
	
	
	/** 
	* @param array
	* @return array */
	public function values($just_these_values=null){
		if(is_null($just_these_values)) $just_these_values = array_keys($this->_values);
		if(!is_array($just_these_values)) return $this->error->should_be_an_array_of_nonempty_strings($just_these_values);
				
		$values = array();
		//could do array_intersect_key, but that wouldn't preserve the order of $just_these_values
		foreach($just_these_values as $field){
			if(empty($field) || !is_string($field)) return $this->error->should_be_an_array_of_nonempty_strings($just_these_values);
			if(array_key_exists($field, $this->_values)){
				$values[$field] = $this->$field;
			}
		}
		
		return $values;
	}	
	
	/**
	* @param array
	* @return boolean
	*/
	public function set_values($values, $offset = 0){
		if(!is_array($values)) return $this->error->should_be_an_array($values, $offset+1);
		foreach($values as $field => $value){
			if($this->field_is_readonly($field)) return $this->error->should_be_a_writeable_field($field);
			$this->$field = $value;
		}
		return true;
	}
	
	public function load_field_values_from_db(){
		$id = $this->id();
		if(empty($id)) return $this->error->warning("I can't load values from the database for a ".get_class($this).' entity without an id', 1);

		$entity = static::find_one($id);
		if(!static::is_an_entity($entity)) return $this->error->should_be_an_x($id, 'id for a '.get_class($this).' entity', 1);
		$success = $this->_set_field_values($entity->values());
		if($success) $this->_altered_fields = array();
		return $success;
	}	
	
	/**
	* Set multiple trusted values at once - unlike the public equivalent, this method will not validate the values, and should only be used internally when loading values directly from the database.
	* Note that these values will *not* show up in {@link _altered_fields}.
	* @param array
	* @return boolean 	*/
	protected function _set_field_values($values){
		if(array_key_exists('__daas_row_number', $values))
			unset($values['__daas_row_number']); //for offsets, we generate a row number that we don't want to include in the values.
		$this->_values = $values;
		foreach($this->_values as $field => $value){						
			//when using find(), you may alias a column or return a value that isn't a column on the table
			//to accomodate that, we let that be set as a readonly field on the entity - you won't be able to write to it, but it does still exist	
			$class = get_called_class();
			if(!$class::field_exists($field)){
				$this->_readonly_fields[] = $field;
			}
		}
		return true; 
	}
	
	//internal - call this whenever a value is changed from outside the class.
	//do NOT set override_validation to true without careful thought - this should be used only for special circumstances like setting the default value of read-only field
	protected function _set_field_value($field, $value, $error_offset = 0, $override_validation = FALSE){
		if(!$override_validation){
			if(!$this->field_is_writeable($field)) return $this->error->should_be_a_writeable_property($field, $error_offset+1);
			if($this->property_has_validation($field, $value) && !$this->value_is_valid_for_property($field, $value)){
				$field_should_be = 'property_value_should_be_a_'.str_replace('|', '_or_', element($field, $this->property_validation_rules()));
        	    return $this->error->$field_should_be($field, get_class($this), $value, $error_offset+1);
			}
		}
		
		$this->_values[$field] = $value;
		$this->_altered_fields[] = $field;
		return true;
	}
		
		
	/** 
	* Fields which cannot be written to the database (at least by the developer).
	*
	* This will generally include the primary key, and any fields which do not correlate to columns on the database table (e.g., calculations or joined table columns
	* that were pulled from the database and made available on this entity.)
	*
	* Format: array(column_name_one, column_name_2)
	*
	* @return array 
	*/	
	public function readonly_fields(){
#TODO - NEED TO EXCLUDE ANYTHING WHICH ISN'T ON THE TABLE		
		return $this->_merge_with_parent_array('readonly_fields');
	}
	
	/** 
	* @param string
	* @return boolean */
	public function field_is_readonly($field){
		return in_array($field, $this->readonly_fields());
	}
	
	/**
	* @param string
	* @return boolean */
	public function field_is_writeable($field){
		return $this->field_exists($field) && !$this->field_is_readonly($field);
	}
	
	/** return array */
	public function writeable_fields(){
		return array_diff(static::fields(), $this->readonly_fields());
	}
	
	/** return array */
	public function writeable_values(){
		return array_intersect_key($this->values(), array_flip($this->writeable_fields()));
	}

	/**
	* @param array
	* @return boolean 	*/
	public function save(){		
			
		//create or update the record
		$primary_key_value = $this->id();
		#if(empty($primary_key_value) || !static::exists($primary_key_value)) //why did I have exists here?  it's an extra db query and we'll get a db error anyway if it doesn't exist
		if(empty($primary_key_value))
			$success = $this->_create();
		else			
			$success = $this->_update();
		
		if($success) $this->_altered_fields = array(); //we've now reloaded all the values, so we need to reset the altered fields trackere
		return $success;
	}
	
	/**
	* @param array
	* @return string
	*/
	public function to_json( $just_these_values=null ){
		return json_encode($this->values( $just_these_values ));
	}
	
	///////////////////////////////
	// DATA MANAGEMENT (INSTANCE)
	//////////////////////////////
	
	
	protected function _create(){		
		$error_message = 'Unable to create new '.$this->model_alias;
		
		if(!$this->_run_before_create() || !$this->_run_before_create_and_update()) 
			return $this->error->warning($error_message, 2); //run any specific actions for the child class; quit if actions fail.
		if(!$this->_values_are_valid_for_create() || !$this->_values_are_valid_for_create_and_update()){
			return $this->error->warning($error_message, 2);
		}
		
		//for create, save all writeable values - they may have defaults specified, even if we haven't altered anything
        $success = $this->db()->insert(static::$table, $this->_values_for_save()); 
        if(!$success) return false;
        $id = $this->db()->insert_id();
		if(!static::formatted_like_an_id($id)) return $this->error->warning($error_message, 2);
		$this->_values[static::$primary_key] = $id; //usually you should use _set_field_value - _create() is the only place where we set this manually
		
		//reload the values in case the database automatically generates any values (timestamps, etc.)	
		$this->load_field_values_from_db();
		
		if(!$this->_run_after_create()) return false; //if any cleanup (deletions) or error messages need to happen on failure, run_after_create should supply them
		if(!$this->_run_after_create_and_update()) return false;
		return true;
	}
	
	protected function _update(){
		$error_message = 'Unable to update '.$this->error->describe($this).' with the following values: '.$this->error->describe($this->_values_for_save());
		if(!$this->_run_before_update() || !$this->_run_before_create_and_update())
			return $this->error->warning($error_message);		
		if(!$this->_values_are_valid_for_update() || !$this->_values_are_valid_for_create_and_update()){
			return $this->error->warning($error_message);
		}
		
		//for update, check to make sure that we've set some of the values to make sure that we really need to save
		$values_for_save = $this->_values_for_save();
		if(empty($values_for_save)) return true; //the update is unnecessary, so return true (but not false or with a notice, because it's not an invalid behavior to save an entity that hasn't changed)
		
		$success = $this->db()->update(static::$table, $values_for_save, array(static::$primary_key => $this->id()));	
		if(!$success) return $this->error->warning($error_message);
		
		//reload the values in case the database automatically generates any values (timestamps, etc.)	
		$this->load_field_values_from_db();			

		if(!$this->_run_after_update()) return false;
		if(!$this->_run_after_create_and_update()) return false; 
		
		return true;
	}
	
	protected function _values_for_save(){
		if(!isset($this->id)){
			return array_merge($this->writeable_values(), $this->altered_values()); //do all writeable values, so that we grab any default values when necessary.  include altered values in case we've programmatically altered some read-only values
		}
		return $this->altered_values();
	}

	/**#@+
	* @return boolean
	*/
	protected function _run_before_create(){ 
		$writeable_values = $this->_values_for_save();
		if(empty($writeable_values)){
			return $this->error->warning('Unable to create a new '.$this->model_alias.' without any writeable values.  Current values: '.$this->error->describe($this->values()));
		}
		return true; 
	}
	protected function _run_before_update(){ return true; }
	protected function _run_before_create_and_update(){ return true; }
	
	protected function _values_are_valid_for_create(){ return true; }
	protected function _values_are_valid_for_update(){ return true; }
#TODO - CONSIDER CHECKING MAXLENGTH, NULL VALUES
	protected function _values_are_valid_for_create_and_update(){ return true; }
	
	protected function _run_after_create(){ return true; }
	protected function _run_after_update(){ return true; }
	protected function _run_after_create_and_update(){ return true;}		
	/**#@-*/ //end phpdoc template for _run_before/_run_after methods
	
	///////////////////////////////
	// RELATIONSHIPS (INSTANCE)
	///////////////////////////////
	
	/**
	* @param string
	* @param array
	* @return boolean
	*/
	public function has_related($relationship, $additional_conditions=array()){
		return ($this->count_related($relationship, $additional_conditions) >= 1);
	}

	
	/**
	* @param string
	* @param array
	* @return int
	*/
	public function count_related($relationship, $additional_conditions=array()){
		$relationship = singular($relationship);              
        $success = $this->set_up_relationship_query($relationship, $additional_conditions);
		if(!$success) return array();
		
		//MG - if you got this far, you know the relationship is valid, you don't need to check for it -- MG
		$related_model = static::related_model($relationship);
		return $related_model::count();
	}
	
	/**
	* @param string
	* @param array
	* @param string
	* @return array
	*/
	public function find_related($relationship, $additional_conditions=array(), $key_results_by = null){ 
		$relationship = singular($relationship);              
        $success = $this->set_up_relationship_query($relationship, $additional_conditions);
		if(!$success) return array();
		
		//MG - if you got this far, you know the relationship is valid, you don't need to check for it -- MG
		$related_model = static::related_model($relationship);
		$type = static::relationships($relationship, 'type');
		$related_model::db()->select($related_model::$table.'.*', $add_backticks = FALSE);	//limit the resulting columns to just the related model		
		if($type == 'belongs_to')
			return $related_model::find_one();
		else
			return $related_model::find(array(), $key_results_by);
    }	
	
	/**
	* @param string
	* @param array
	* @return boolean 	*/
	public function set_up_relationship_query($relationship, $additional_conditions=array()){
		
		//make sure that we're dealing with a valid relationship
		$relationship = singular($relationship);
		if(!static::has_relationship($relationship)) return $this->error->relationship_does_not_exist($relationship, $this->model_alias);
				
		//find the key relationship information
		$related_model = static::related_model($relationship);
		$related_db = $related_model::db();
		$relationship_info = static::relationships($relationship);
		
		$type = element('type', $relationship_info);
		if($type == 'has_and_belongs_to_many'){
			//has_and_belongs_to_many has to be a relationship join, not a simple query
			$related_model::db()->select( $related_model::$table.'.*' );
			return $this->set_up_relationship_join($relationship, $additional_conditions);	
		}	
				
		$foreign_key = static::foreign_key($relationship);  //what this model considers to be the foreign key
		$related_foreign_key = static::related_foreign_key($relationship);	//what the other model considers to be the foreign key?	
		$key = static::key_for_relationship($relationship); //usually the primary key; occasionally another unique key, in weird cases like the users table
		
		$table = static::$table;
		$related_table = $related_model::$table;

		if($type == 'has_many' || $type == 'has_one'){
			$related_db->where($related_table.'.'.$foreign_key, $this->$key);
		}
		elseif($type == 'belongs_to'){ //foreign key is on this table
			$related_db->where($related_table.'.'.$key, $this->$related_foreign_key);
		}
		else
			return $CI->error->should_be_a_known_relationship_type($type);
		
		if(!empty($additional_conditions))
			$related_model::_set_conditions($additional_conditions);
		if(!empty($relationship_info['condition']))
			$related_db->where($relationship_info['condition'], NULL, false); //extra params tell db not to escape this
		if(!empty($relationship_info['order_by']))
			$related_db->order_by($relationship_info['order_by']); 					

		return true; 
	}	

	
	/**
	* @param string
	* @param array
	* @return boolean 	*/
	public function set_up_relationship_join($relationship, $additional_conditions=array(), $join_type=''){	
		if(!is_string($join_type)) return $this->error->should_be_a_string($join_type);
		if(!empty($join_type) && !in_array(strtoupper($join_type),  array('LEFT', 'RIGHT', 'OUTER', 'INNER', 'LEFT OUTER', 'RIGHT OUTER')))
			return $this->error->should_be_a_known_join_type($join_type);
		
		//make sure that we're dealing with a valid relationship
		$relationship = singular($relationship);
		if(!static::has_relationship($relationship)) return $this->error->relationship_does_not_exist($relationship, $this->model_alias);
		
		//find the key relationship information
		$related_model = static::related_model($relationship);
		$related_db = $related_model::db();
		$relationship_info = static::relationships($relationship);
			
		$foreign_key = static::foreign_key($relationship, $include_table=true);  //what this model considers to be the foreign key
		$related_foreign_key = static::related_foreign_key($relationship);	//what the other model considers to be the foreign key?	
		$key = static::key_for_relationship($relationship); //usually the primary key; occasionally another unique key, in weird cases like the users table

		//if the related model is in another database, we'll be selecting from that database.  so this table's tablename should include the database name.
		$table = static::$table;
		$db = static::db(); //unnesc. step, but makes phpdocumentor happier
		if($db->database != $related_db->database){
			$table = $db->database.'.dbo.'.$table;
		}
		
		$type = element('type', $relationship_info);

		if($type == 'has_many' || $type == 'has_one'){
			$related_db->join($table, $foreign_key.' = '.$table.'.'.$key, $join_type);
			$related_db->where($table.'.'.$key, $this->$key);
		}
		elseif($type == 'belongs_to'){ //foreign key is on this table
			$related_db->join($table, $table.'.'.$related_foreign_key.' = '.$related_model::$table.'.'.$key, $join_type);
			$related_db->where($table.'.'.$related_foreign_key, $this->$related_foreign_key);
		}
		elseif($type == 'has_and_belongs_to_many'){ //foreign keys are on a relationship table
			#TODO - WILL THROUGH WORK WITH MULTIPLE DBS?  MIGHT BE OK AS LONG AS THE DATABASE IS SPECIFIED IN THE RELATIONSHIP CONFIG
			$through_table = element('through', $relationship_info); //has_relationship() will verify that this exists and is a valid table
			$related_db->join($through_table, $related_model::$table.'.'.$related_model::$primary_key.' = '.$through_table.'.'.$related_foreign_key, $join_type);
			$related_db->where($through_table.'.'.$foreign_key, $this->$key);
		}
		else
			return $CI->error->should_be_a_known_relationship_type($type);
		
		if(!empty($additional_conditions))
			$related_model::_set_conditions($additional_conditions);
		if(!empty($relationship_info['condition']))
			$related_db->where($relationship_info['condition'], NULL, false); //extra params tell db not to escape this
		if(!empty($relationship_info['order_by']))
			$related_db->order_by($relationship_info['order_by']); 					

		return true; 
	}
	
///////////////////////////
// GETTERS
///////////////////////////

	/**
	* Array containing the validation rules for the class variables of this object.
	* This is a getter for {@link _property_validation_rules}.
	*
	* @uses _property_validation_rules
	* @uses _merge_with_parent_array
	*
	* @return array
	*/
	public function property_validation_rules(){	
		return $this->_merge_with_parent_array('property_validation_rules');
	}	
	
//////////////////////////////////////////
// DEALING WITH PROPERTIES / CLASS VARS
//////////////////////////////////////////
	
	/**
	* True if this class has a variable of this name.
	* This method will check both for a variable named $property and for a protected variable named "_{$property}".
	*
	* @param string Name of the class variable
	* @param mixed (Optional) Either an object or the name of a class.  Defaults to the current class.
	* @return boolean
	*/
	function property_exists($property, $object_or_class_name = ''){
		if($object_or_class_name === '')
			$object_or_class_name = get_class($this);
			
		return (property_exists($object_or_class_name, $property) || property_exists($object_or_class_name, '_'.$property) || array_key_exists($property, $this->_values) || static::field_exists($property));
	}
	
	//unsets for the purposes of the class by setting to null, but doesn't actually call unset() so that the class still knows the property exists.  (exists -- is not set.)
	function unset_property($property){
		if(!$this->property_exists($property)) return $this->error->property_does_not_exist($property);
		$property_name_possibilities = array('_'.$property, $property);
		foreach($property_name_possibilities as $possibility){
			if(property_exists(get_class($this), $possibility))
				return $this->$possibility = null;
		}
		if(array_key_exists($property, $this->_values))
			$this->_values[$property] = null;
		
		return $this->error->warning("I was unable to unset ".get_class($this).'::'.$property);
	}
	
	/**
	* Returns false if the class var has a non-empty and non-zero value.
	* This method operates based on the same rules as PHP's {@link empty()} function.  Due to the custom __get and __set methods used in this class,
	* empty($this->property) will not always be reliable.  This method takes into account the special naming rules (property vs _property).  
	*
	* Additionally, sometimes variables tht have a custom getter method will have a default value in the getter method rather than in the variable declaration.  
	* This method takes that into account.
	*
	* @param string Name of the class variable
	* @return boolean True if it has an empty or zero value
	*/
	function property_is_empty($property){
		if(!$this->property_exists($property)) return $this->error->property_does_not_exist($property, get_class($this));
		#if(!isset($this->$property)) return true; //tempting, but sometimes we put the default value into the method, not into the class var.
		if(property_exists(get_class($this), $property)) return empty($this->$property);
		if(method_exists(get_class($this), $property)){
			$value = $this->$property();
			return empty($value);
		}
		if(static::field_exists($property) || array_key_exists($property, $this->_values)) return empty($this->_values[$property]);
		$internal_property = '_'.$property;
		if(property_exists(get_class($this), $internal_property))
			return empty($this->$internal_property); 
		
		
		$value = $this->$property;		
		return empty($value);
	}
	
	/**
	* Returns true if a property validation rule is set up for this property.
	*
	* @uses property_validation_rules
	*
	* @param string Name of a class var
	* @return boolean
	*/
	public function property_has_validation($property){
		if(!$this->property_exists($property)) return $this->error->property_does_not_exist($property, get_class($this), 1);
		
		return array_key_exists($property, $this->property_validation_rules);
	}
	
	/**
	* Checks a potential value for property against the validation rule for that property.
	*
	* @uses _property_validation_rules
	*
	* @param string Name of a class var
	* @param mixed Value for the class var
	* @return boolean
	*/
	public function value_is_valid_for_property($property, $value){		
		if(!$this->property_has_validation($property)) return $this->error->should_be_a_property_with_a_validation_rule($property);
		
		$validation_rule = $this->property_validation_rules[$property];
		return $this->is->$validation_rule($value);
	}
	
	/**
	* Intended to be used for configuration arrays -- a way to extend a parent array rather than completely override it.
	*
	* Normally, 
	*
	*
	* @param string Name of a class var.
	* @return array
	*/
	protected function _merge_with_parent_array($property){
		if(!$this->property_exists($property)) 
			return $this->error->property_does_not_exist($property, get_class($this));
		
		$internal_property = $property;
		if(property_exists($this, '_'.$property))
			$internal_property = '_'.$property;		
				
		if(isset($this->$internal_property) && !is_array($this->$internal_property)) 
			return $this->error->should_be_an_array($this->$internal_property);	
		
		$parent_value = $this->get_parent_default_value($property);
		
		if(!empty($parent_value) && $parent_value !== $this->$internal_property){
			if(!is_array($parent_value)){
				$message = "I can't merge ".get_class($this)."->".$property." with its parent value because parent::".$property." is not an array";
				trigger_error($message, $this->level('should_be_an_array'));
			}
			elseif(empty($this->$internal_property))
				return $parent_value;
			else
				return array_merge($parent_value, $this->$internal_property);	
		}
		
		return $this->$internal_property;
	}

	/**
	* Get the default value of a class variable in the parent class.
	* This function will ignore the value of the variable in the current class and get the default value of it for the parent class.
	*
	* <code>
	* class Mother extends Object{
	* 	var	$name = 'Molly';
	* }
	*
	* class Daughter extends Mother{
	* 	var $name = 'Ginny';
	*	}
	*
	* $daughter = new Daughter();
	* echo $daughter->name; //echoes 'Ginny';
	* echo $daughter->get_parent_default_value('name'); //echoes 'Molly'
	*
	* $mother = new Mother();
	* echo $mother->name; //echoes 'Molly';
	* echo $mother->get_parent_default_value('name'); //void, since Object doesn't have a property called 'name'
	* </code>
	*
	* Like most of the methods in this class, this method will check for both a variable named $property and for a protected variable named "_{$property}".
	*
	* @param string Name of the class variable
	* @return mixed 
	*/
	protected function get_parent_default_value($property, $object_or_class_name = null){
		if(is_null($object_or_class_name)) 
			$object_or_class_name = get_class($this);//note -- get_parent_class($this) would always consider $this to be the Object class, not any child classes
		
		$parent_class = get_parent_class($object_or_class_name); 
		if(!empty($parent_class) && $this->property_exists($property, $parent_class)){
			$parent = new $parent_class();
			return $parent->$property;			
		}
	}	
	
	//uses func_get_args to allow a variable number of parameters
	function method_return_value_is_empty($method){
		if(!$this->is->is_a_nonempty_string($method) || !method_exists(get_class($this), $method))
			return $this->error->method_does_not_exist($method, get_class($this));
		
		$args = func_get_args();
		unset($args[0]); //remove the method name from the function args
		
		$return_value = call_user_func_array(array($this, $method), $args);
		return empty($return_value);	
	}
	
	static function default_value_for_property($property, $class){
		$instance = new $class();
		if(!$instance->property_exists($property)) return $instance->error->property_does_not_exist($property,  $class);
		return $instance->$property;
	}					
	
	///////////////////////////////////////////////////////////////////////////////////////////////////
	// MAGICAL FUNCTIONS THAT MARGARET MAY REGRET INCLUDING
	///////////////////////////////////////////////////////////////////////////////////////////////////
	
	/**
	*  DO NOT ACCESS THIS FUNCTION DIRECTLY.
	*  This overrides a magical PHP function that's automatically called when someone tries to access a property that doesn't exist. We use this to provide access to the 
	*  database columns and related entities for a given entity.
	*
	*  For example, if you have a section entity that has a title column and a description column in the database, you can access these values:
	*  		$section->title
	* 		$section->description
	*
	*  You can get an array of all of the columns for this entity by calling $section->values.
	*
	*  You can also access the related entities.  If section has a belongs_to relationship with survey, and a has_many relationship with question:
	* 		$section->survey 	//returns the related survey entity
	* 		$section->questions	//returns an array of the related question entities
	* 
	*  @see http://us3.php.net/manual/en/language.oop5.overloading.php#language.oop5.overloading.members
	*/
	function __get($property){
		$class = get_called_class(); //doing this just to make phpdocumentor happier with it - can switch back to using static if we upgrade phpdocumentor
		
		$indirect_property = '_'.$property;	
		if(method_exists(get_class($this), $property))
			return $this->$property();	
		elseif(property_exists(get_class($this), $indirect_property))
			return $this->$indirect_property;	
		elseif(property_exists(get_class($this), $property))
			return $this->$property; //we're hitting this because we're out of scope, but we don't mind giving read-only access
		elseif(array_key_exists($property, $this->_values))
			return $this->_values[$property];
		elseif($class::field_exists($property)){
			if($property != $class::$primary_key && $property != 'id' && isset($this->id)) $this->error->warning(get_class($this).'::$'.$property.' was not loaded from the database', 1);
			return null; //it wasn't in $this->_values, so we don't have a value for it at the moment.			
		}
		elseif($class::has_relationship(singular($property)))
			return $this->find_related(singular($property));
		elseif(string_ends_with('_count', $property) && $class::has_relationship(strip_from_end('_count', $property)))
			return $this->count_related(singular(strip_from_end('_count', $property)));
		
		$this->error->property_does_not_exist($property, get_class($this), 1);
		return null;
	}
	
	function __isset($property){		
		$class = get_called_class(); //doing this just to make phpdocumentor happier with it - can switch back to using static if we upgrade phpdocumentor
	
		if(property_exists(get_class($this), $property))
			return isset($this->$property);
		
		$indirect_property = '_'.$property;		
		if(property_exists(get_class($this), $indirect_property))
			return isset($this->$indirect_property);
	
		//id is a special case
		if($property == 'id' && !$class::field_exists($property)){
			$property = $class::$primary_key;
		}		
		
		return isset($this->_values[$property]);
	}	
	
	/** 
	*  @see http://us3.php.net/manual/en/language.oop5.overloading.php#language.oop5.overloading.members
	*/
	function __set($property, $value){
		$table = static::$table; //not nesc., but makes phpdocumentor happier
					
		//if there's a set_property method set up for this property, use it
		$setter_method = 'set_'.$property;
		if(method_exists($this, $setter_method)){
			//ok to have a setter in the current class && a validator in the parent class, but not defined in the same class.
			if(method_exists($this, $setter_method) && $this->property_has_validation($property) && array_key_exists($property, $this->_property_validation_rules)){
				$this->error->notice('You have both a method called set_'.$property.' and a validation entry for '.get_class($this).'->'.$property.'. Only the method will be used'); 		
			}
			return $this->$setter_method($value, 1);
		}
		
		//if it's a field for the table, set it if it's writeable
		if($this->field_is_readonly($property))
			return $this->error->notice(get_class($this).'::$'.$property.' is a read-only property and may not be set to '.$this->error->describe($value), $offset=1);
		elseif(static::field_exists($property))
			return $this->_set_field_value($property, $value, 1);
				
		//if there's a property validation rule set up, use it
		if($this->property_has_validation($property)){			
			if($this->value_is_valid_for_property($property, $value)){
				$internal_property = $property;
				if(!property_exists(get_class($this), $property))
					$internal_property = '_'.$property;
				return $this->$internal_property = $value;
			}
			else{
				$property_should_be = 'property_value_should_be_a_'.str_replace('|', '_or_', element($property, $this->property_validation_rules()));
				return $this->error->$property_should_be($property, get_class($this), $value, 1);
			}
	
		}
		
			
		//if there's no such property (public OR private OR protected, etc), let the developer know that they've made a mistake	
		if(!$this->property_exists($property)){
			return $this->error->property_does_not_exist($property, get_class($this), 1);
		}
		
		//if the property exists, then it must not be in scope (otherwise we wouldn't have gotten to this function
		return $this->error->warning('The property "'.$property.'" exists for the '.get_class($this).' class, but it is not in scope');			
		
	}
	
	/**
	*  DO NOT ACCESS THIS FUNCTION DIRECTLY.
	*  This overrides a magical PHP function that's automatically called when someone tries to access a method that doesn't exist. We use this to provide access to the 
	*  model methods for a given entity.
	*
	*  For example, if your model has a method called dance_the_hokey_pokey that takes ($right_foot, $left_foot) as parameters, you can either call on it the normal way,
	*  via the model, or if you have an entity declared, you can call it this way:
	*  <code> $entity->dance_the_hokey_pokey($right_foot, $left_foot); </code>
	*  and the entity class will pass the $right_foot, $left_foot params to the model method.
	*
	*  @see http://us3.php.net/manual/en/language.oop5.overloading.php#language.oop5.overloading.members
	*/
	function __call($name, $arguments){
		$class = get_called_class(); //doing this just to make phpdocumentor happier with it - can switch back to using static if we upgrade phpdocumentor
		if($class::has_relationship($name)){
			$arguments = array_merge(array(singular($name)), $arguments);
			return call_user_func_array(array($this, 'find_related'), $arguments);
		}
		elseif(string_ends_with('_count', $name) && $class::has_relationship(strip_from_end('_count', $name))){	
			$arguments = array_merge(array(singular(strip_from_end('_count', $name))), $arguments);
			return call_user_func_array(array($this, 'count_related'), $arguments);
		}
		elseif(string_begins_with('has_', $name)){
			$relationship = strip_from_beginning('has_', $name);			
			if($class::has_relationship($relationship)){
				$arguments = array_merge(array($relationship), $arguments);
				return call_user_func_array(array($this, 'has_related'), $arguments);
			}
		}
					
		$this->error->warning('There is no method called '.$name.' for '.get_class($this), 1);
		return false;	
	}	

	
///////////////////////////////////////////////////////////////////////////////////////////////////
// STATIC METHODS
// Methods concerning the table that powers this model, or multiple records in that table.
///////////////////////////////////////////////////////////////////////////////////////////////////
	
	
	/////////////////////////////////
	// Table Metadata
	/////////////////////////////////
	
	/**
	* True if the field exists in the table powering this model.
	* @param string
	* @return boolean
	*/
	public static function field_exists($field){
		return in_array($field, static::fields());
		
#		$table = static::$table; 
#		if(empty($table)) return false; //in case this gets called from the entity class
#		return static::db()->field_exists($field, $table);
	} 
	
	/**
	* Field names for the table powering this model.
	* Note that this makes use of a CI method which caches the response, so you may run this method multiple times without quereying the database multiple times.
	* @return array
	*/
	public static function fields(){
		if(!empty(static::$table))
			return static::db()->list_fields(static::$table);	
		return array();
	}	
	
	/** 
	* True if a value is formatted like a valid value for the primary key of the table powering this model.
	* @param mixed
	* @return boolean	
	*/
	public static function formatted_like_an_id($id){
		return validates_as('nonzero_unsigned_integer', $id);
	}
	
	/** 
	* True if a values is an entity of this model
	* @param mixed
	* @return boolean 
	*/
	public static function is_an_entity($entity){
		return is_a($entity, get_called_class());
	}	

	/////////////////////////////////
	// DATA MANAGEMENT (STATIC)
	/////////////////////////////////	

	public static function create($values){
		//verify that we have an array
		if(!is_array($values)) return get_instance()->error->should_be_an_array($values);
		
		$class = get_called_class();
		$entity = new $class($values);
		
		if(!$entity->save()) return false; //save() will trigger any necessary errors
		return $entity;
	}

	
	public static function update($id, $values){ 
		if(!static::formatted_like_an_id($id)) return get_instance()->error->should_be_an_id($id);
		if(!is_array($values) || empty($values)) return get_instance()->error->should_be_an_array($values);
		
		$entity = static::find_one($id);	
		if(!static::is_an_entity($entity)) return get_instance()->error->should_be_an_x('id for an existing '.static::model_alias().' entity', $id);
		
		//set the new values and verify that all of the values are writeable
		//note that this may need to h
		$entity->set_values($values);
		if(!$entity->save()){
			return false;
		}
		return $entity;
	}
	
	public static function delete($id){ 
		$class = get_called_class(); //doing this just to make phpdocumentor happier with it - can switch back to using static if we upgrade phpdocumentor 
		if(!$class::formatted_like_an_id($id)) return get_instance()->error->should_be_an_id($id);
		if(!$class::_db_config_is_valid()) return false; 
		$entity = $class::find_one($id);		
		if(!$class::is_an_entity($entity)) return get_instance()->error->should_be_an_x('id for an existing '.$class::model_alias().' entity', $id);
			
		if(!$class::_run_before_delete($entity)) return false;
		if(!$class::_delete($id)){
			return get_instance()->error->warning('Unable to delete '.get_class($entity).'#'.$id);
		}
		$class::_run_after_delete($entity);
		return true; 
	} 
	
	protected static function _run_before_delete(&$entity){ return true; }
	protected static function _delete($id){ 
		$table = static::$table; //unnesc., but makes phpdocumentor happier
		$primary_key = static::$primary_key;
		return static::db()->where($primary_key, $id)->delete($table); 
	}
	protected static function _run_after_delete($entity){}	
		
	/////////////////////////////////
	// SEARCH
	/////////////////////////////////

	/**
	* Performs a database search and returns an array of entities that that match the given id/conditions.
	* 
	* This method makes use of CI's active record features, which allows the developer to layer on joins, 
	* conditions, etc., before calling this method.  To easily view the output of this method, turn on the
	* CI Profiler.   
	*
	* To search for only one entity, use {@link find_one()}
	*
	* @param int|array The id of an entity or an array of conditions, formatted array(column => value)
	* @param string Database column that should be used as the key for the return value.
	* @return array Either a single {@link Entity} or an array of {@link Entity} objects
	*/		
	public static function find($id_or_conditions = array(), $key_by = null){
		$CI = get_instance();
		$table = static::$table; //unnesc., but makes phpdocumentor happier
		$primary_key = static::$primary_key;		
		
		//validate the input
		if(is_null($key_by)) $key_by = $primary_key;
		if(!$CI->is->nonempty_string($key_by)) return $CI->error->should_be_a_nonempty_string($key); //note that it's ok for the key to not be a known field if it's manually added it to the select
		if(!static::_set_conditions($id_or_conditions)) return array(); //_set_conditions will validate conditions & trigger error as necessary
		
		$results = static::db()->get($table)->result_array();
		if(empty($results)) return $results;	
		
		//check to see make sure that the key that we're meant to be using is meaningful - if not, default to the primary key if it's in these results
		if(!is_string($key_by) || !array_key_exists($key_by, first_element($results))){
			if(array_key_exists($primary_key, first_element($results))){
				$CI->error->should_be_a_column_for_table($key_by, $table);
				$key_by = $primary_key;
			}else{
				$CI->error->warning($CI->error->describe($key_by).' is not a known key.  The resulting array will not have meaningful keys.');
				return $results;
			}
		}		

		$entities = array();
		$entity_class = get_called_class();
		foreach($results as $key => $result){
			$entity = new $entity_class();
			$entity->_set_field_values($result);
			$entities[$result[$key_by]] = $entity;
		}
	
		return $entities; 
	}
	
	/**
	* Performs a database search and returns an entity which matches the given id/conditions.
	*
	* If multiple records match the search criteria, this method will return the first entity that matches
	* the criteria.  To return all entities that match the criteria, use {@link find()}.
	*
	* @param int|array The id of an entity or an array of conditions, formatted array(column => value)
	* @return Entity|false 
	*/		
	public static function find_one($id_or_conditions=array()){
		static::db()->limit(1);
		return first_element(static::find($id_or_conditions));
	}

	/**
	* True if a record exists that matches the id or conditions provided.
	* This will run a count rather than a select, so it's a bit easier on the database then using {@link find} if you don't need the resulting entities.
	* @param int|array The id of an entity or an array of conditions, formatted array(column => value)
    * @return boolean 
	*/
	public static function exists($id_or_conditions = array()){
		return (static::count($id_or_conditions) >= 1);
	}		
	
	/**
	* Returns the number of records that match the id or conditions provided.
	* @param int|array The id of an entity or an array of conditions, formatted array(column => value)
    * @return int 
	*/
	public static function count($id_or_conditions = array()){
		if(!static::_set_conditions($id_or_conditions)) return false;
		return static::db()->count_all_results(static::$table);
	}
	
	/**
	* Returns an array of the number of rows that match the given conditions, grouped by the provided field
	* Returns the number of records that match the id or conditions provided.
	* @param string The name of the field to group by
	* @param int|array The id of an entity or an array of conditions, formatted array(column => value)
    * @return array Formatted array( field_value => count)
	*/
	public static function count_by($field, $id_or_conditions=array()){
		if(!static::field_exists($field)) return get_instance()->error->should_be_a_column_for_table($field, static::$table);
		if(!static::_set_conditions($id_or_conditions)) return false;
				
		$count_field = '__count_'.$field;
		static::db()->group_by($field);
		static::db()->select($field.', COUNT('.$field.') AS '.$count_field);
		$results = static::db()->get(static::$table)->result_array();
		if(empty($results)) return array();
		
		//format the results
		$formatted_results = array();
		foreach($results as $key => $row){
			$formatted_results[$row[$field]] = $row[$count_field];
		}	
			
		return $formatted_results;
	}
	
	
	/**
	* @param int|string|array
	* @return boolean
	*/
	protected static function _set_conditions($id_or_conditions=array()){		
		if(!static::_db_config_is_valid()) return false;
		
		$CI = get_instance();
	
		if(empty($id_or_conditions)){
			if(is_array($id_or_conditions) || static::formatted_like_an_id($id_or_conditions)) 
				return true; //param was set properly, we're just not setting any conditions
			else $CI->error->should_be_an_id_or_search_criteria($id_or_conditions);
		}
		
		//searching by id
		if(static::formatted_like_an_id($id_or_conditions)) 
			return static::db()->where(static::$primary_key, $id_or_conditions); 
		
		//searching by valid CI conditions
		if((!is_numeric($id_or_conditions) && is_string($id_or_conditions)) || $CI->is->nonempty_array($id_or_conditions))
			return static::db()->where($id_or_conditions);
		
		//invalid parameter was passed	
		return $CI->error->should_be_an_id_or_a_string_or_an_associative_array($id_or_conditions);
	}	

	
	/*
	* Checks to make sure we have a valid database connection, table, and primary key for this model.
	* This method should be used to check the database connection before running database queries, so that we'll get useful error messages if the configuration is invalid.
	* @return boolean
	*/
	protected static function _db_config_is_valid(){
		$db = static::db();

		//make sure the database established a connection	
		if(!is_subclass_of($db, 'CI_DB') /*|| !is_resource($db->conn_id) */) //for some reason, this doesn't always populate  
			return get_instance()->error->warning(get_called_class().' could not establish a connection for the '.get_instance()->error->describe(static::$database_group).' database group');
			
		//make sure that table exists for this connection - note that table_exists caches its results and will not query the db more than once no matter how many times we call it	
		if(!$db->table_exists(static::$table)) 
			return get_instance()->error->warning('The '.$db->database.' database does not have a table called '.get_instance()->error->describe(static::$table));
			
		//make sure that table exists for this connection - note that field_exists caches its results and will not query the db more than once no matter how many times we call it		
		if(!static::field_exists(static::$primary_key, static::$table)) 
			return get_instance()->error->warning('The '.$db->database.'.'.static::$table.' table does not have a field called '.get_instance()->error->describe(static::$primary_key));
			
		return true; 
	}
	
	/*
	* The description of this model that will be used to refer to it in error messages, labels, etc.
	* @return string
	*/
	protected static function model_alias(){
		return strtolower(get_called_class());
	}
	
	
	
	/** 
	* Retrieves the database connection for this model.
	* Since we're using multiple databases in this application, it will be important to use the correct database connection when running queries.
	* @return object 
	*/
	public static function db(){
		
		$CI = get_instance();
		$database_group = static::$database_group;
		
		if($database_group == 'default'){ return $CI->db; }	
		//if we need to access a non-default db ...		
		if(!isset($CI->$database_group)){
			$CI->$database_group = $CI->load->database($database_group, TRUE);
		}
			
		//note -- default database will remain loaded under $this->db if it's already been loaded	
		return $CI->$database_group;	
	}	
	
	///////////////////////////
	// RELATIONSHIPS (STATIC)
	///////////////////////////
	
#TODO - DOCUMENT RELATIONSHIP TYPES, THROUGH TABLES, ETC. -- ALL OF THE RELATIONSHIP REQUIREMENTS
#TODO - POSSIBLY MAKE THIS FUNSTEOR VROTECTED AGAIN	
#things to specify:
	#type -> has_one, has_many, belongs_to, has_and_belongs_to_many
	#foreign key
	#related foreign key
	#model
	#through 
	#condition
	#order_by	
	/**
	* @param string
	* @param string
	* @return mixed
	*/
	public static function relationships($relationship = null, $field = null){
		$relationships = static::merge_with_parent_array('_relationships');
		
		//if we're looking for all the relationships, return them
		if(empty($relationship)) return $relationships;
		
		//for a specific relationship, make sure that the relationship exists
		if(!is_string($relationship) || !static::has_relationship($relationship)) 
			return get_instance()->error->relationship_does_not_exist($relationship, static::model_alias());
		
		$relationship = singular($relationship);
		if(empty($field)) return $relationships[$relationship];
		
		if($field == 'model') return element($field, $relationships[$relationship], $relationship);
		return element($field, $relationships[$relationship]);
	}
	
	/**
	* Returns the class name of the model for a given relationship.
	* @param string
	* @return string
	*/
	public static function related_model($relationship){
		if(!static::has_relationship($relationship)) return get_instance()->error->relationship_does_not_exist($relationship, static::model_alias());
		return ucfirst(static::relationships($relationship, 'model'));
	}
	
#TODO - ENFORCE FOREIGN KEYS FOR TYPE?	
	//for relationships where the foreign key field contains a unique identifier for an entity of this model
	/**
	* @param string 
	* @param boolean
	* @return string
	*/
	public static function foreign_key($relationship='', $include_table=false){	
		if(!is_string($relationship)) return get_instance()->should_be_a_string($relationship);
		if(!is_bool($include_table)) return get_instance()->should_be_a_boolean($include_table);
	
		$table = static::$table; //unnesc., but makes phpdocumentor happier
		$primary_key = static::$primary_key;
		//if no relationship is specified, go with the standard pattern
		if(empty($relationship)){
			if($primary_key == 'id')
				$foreign_key = static::model_alias().'_'.$primary_key;
			else	
				$foreign_key = $primary_key;	
				
			if($include_table) $foreign_key = $table.'.'.$foreign_key;
			return $foreign_key;	
		}
		
		//if the relationship is specified, check relationship config for a foreign key
		if(!static::has_relationship($relationship)) return get_instance()->error->relationship_does_not_exist($relationship, static::model_alias());
		$relationship_info = static::relationships($relationship);
		$foreign_key = element('foreign_key', $relationship_info, static::foreign_key()); //default to the standard foreign key if one wasn't specified
		if(!$include_table) return $foreign_key;

		//if this relationship doesn't have a through table, the foreign key will be part of the related model's table
		$related_model = static::related_model($relationship);
		$table = element('through', $relationship_info, $related_model::$table);
		return $table.'.'.$foreign_key;
	}
	
#TODO - SWITCH THIS BACK TO RELATED FOREIGN KEY. MAKE SURE DOCBLOCK IS CLEAR THAT THIS IS THE FOREIGN KEY FOR THE RELATED OBJECT.  DRAT.  YOU WILL NEVER EXPRESS THIS WELL.	
	//for relationships where the foreign key field contains a unique identifier for an entity of the *related* model
	/**
	* @param string 
	* @param boolean
	* @return string
	*/
	public static function related_foreign_key($relationship, $include_table = false){
		if(!static::has_relationship($relationship)) return get_instance()->error->relationship_does_not_exist($relationship, static::model_alias());
		if(!is_bool($include_table)) return get_instance()->error->should_be_a_boolean($include_table);
		
		//related_foreign_key may be specified in the relationship config
		$related_foreign_key = static::relationships($relationship, 'related_foreign_key');
		
		//or it may default to the standard foreign key for the related model
		if(empty($related_foreign_key)){
			$related_model = static::related_model($relationship);
			$related_foreign_key = $related_model::foreign_key();
		}
		
		if(!$include_table) return $related_foreign_key;		
		
		//if this relationship doesn't have a through table, the related foreign key will be part of this model's table
		$table = static::relationships($relationship, 'through');
		if(empty($table)) $table = static::$table;
		return $table.'.'.$related_foreign_key;
	}
	
	//unique identifier used in this relationship - usually should be the primary key, may occasionally be something else 
	//for has_one/has_many relationships, this will be a column on this model; for belongs_to relationships, this will be a column on the other model
	/**
	* @param string
	* @param boolean
	* @return string
	*/
	public static function key_for_relationship($relationship, $include_table = false){
		$class = get_called_class(); //doing this just to make phpdocumentor happier with it - can switch back to using static if we upgrade phpdocumentor
		if(!$class::has_relationship($relationship)) return get_instance()->error->relationship_does_not_exist($relationship, $class::model_alias());
		if(!is_bool($include_table)) return get_instance()->error->should_be_a_boolean($include_table);
		
		$table = $class::$table; //unnesc., but makes phpdocumentor happier
		$primary_key = $class::$primary_key;
		//key for relationship may be specified in the relationship config
		$key_for_relationship = $class::relationships($relationship, 'key_for_relationship');
		if(empty($key_for_relationship)){
			if($class::relationships($relationship, 'type') == 'belong_to'){
				$related_model = $class::related_model($relationship);
				$key_for_relationship = $related_model::$primary_key;
			}else
				$key_for_relationship = $primary_key;
		}
		if(!$include_table) return $key_for_relationship;
		return $table.'.'.$key_for_relationship; 
	}


	/**
	* True if the given values is the name of a relationship for this model.
	*
	* @todo optimize - can we move the verification code to a better location?
	* @param string
	* @return boolean
	*/
	public static function has_relationship($relationship){
		$class = get_called_class(); //doing this just to make phpdocumentor happier with it - can switch back to using static if we upgrade phpdocumentor
		if(empty($relationship) || !is_string($relationship)) return get_instance()->error->should_be_a_nonempty_string($relationship);
		$relationship = singular($relationship);		
				
		$relationships = $class::relationships();
		if(empty($relationships) || !array_key_exists($relationship, $relationships)) return false;
		
		//before returning true, make sure that the relationship has been set up correctly
		$relationship_info = $relationships[$relationship];
																										
		//verify that a relationship type has been set
		if(empty($relationship_info['type']))
			return get_instance()->error->warning('Ignoring relationship "'.$relationship.'" for '.$class::model_alias().' because no type has been set for it');
		
		//verify that a valid model has been set, or if we're defaulting to the relationship name, make sure there's a model for it
		$model = element('model', $relationship_info, $relationship);
		if(empty($model) || !class_exists(ucfirst($model))){
			$message = 'Ignoring relationship "'.$relationship.'" for '.$class::model_alias().' because no valid model has been set for it. ';
			$message .= 'If "'.$model.'" is the correct model, please make sure that it is within scope.';
			return get_instance()->error->warning($message);
		}
		
		//verify that a through table has been set if this is a has_and_belongs_to_many table
		if($relationship_info['type'] == 'has_and_belongs_to_many'){
			if(empty($relationship_info['through'])) 
				return get_instance()->error->warning('Ignoring relationship "'.$relationship.'" for '.$class::model_alias().' because no through table has been set for it.');
			if(!$class::db()->table_exists($relationship_info['through'])){
				$message = 'Ignoring relationship "'.$relationship.'" for '.$class::model_alias().' because '.get_instance()->error->describe($relationship_info['through']).' is not a table.';
				return get_instance()->error->warning($message);
			}
		}
			
		return true; //if we've made it thus far, we have a valid relationship.  hurrah!
	}
	
	protected static function merge_with_parent_array($property){
		if(empty($property) || !is_string($property)) return get_instance()->error->should_be_a_nonempty_string($property);
		
		//before we do anything, make sure that this is set in this class - if not, we don't need to proceed
		if(!property_exists(get_called_class(), $property)){ 
			return get_instance()->error->property_does_not_exist($property, get_called_class());
		}
		
		//make sure that the property is an array if it's set in this class
		if(!is_array(static::$$property)) return get_instance()->error->should_be_an_array($property);
		
		//make sure that this class has a parent - if not, just return this class's value.		
		$parent_class = get_parent_class(get_called_class());
		if(empty($parent_class)) return static::$$property;
		
		//make sure that parent class has a value for this property - if not, just return this class's value.	
		$parent_value = $parent_class::$$property;
		if(empty($parent_value)) return static::$$property;
		
		//make sure the parent value is an array 
		if(!is_array($parent_value)){
			get_instance()->error->should_be_an_array($parent_value);
			return static::$$property;
		}
		
		return array_merge($parent_class::merge_with_parent_array($property), static::$$property);	
	}	
	
}
